"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * The Highlighter is the server-side entry points for any tool that wishes to
 * highlight elements in some way in the content document.
 *
 * A little bit of vocabulary:
 * - <something>HighlighterActor classes are the actors that can be used from
 *   the client. They do very little else than instantiate a given
 *   <something>Highlighter and use it to highlight elements.
 * - <something>Highlighter classes aren't actors, they're just JS classes that
 *   know how to create and attach the actual highlighter elements on top of the
 *   content
 *
 * The most used highlighter actor is the HighlighterActor which can be
 * conveniently retrieved via the InspectorActor's 'getHighlighter' method.
 * The InspectorActor will always return the same instance of
 * HighlighterActor if asked several times and this instance is used in the
 * toolbox to highlighter elements's box-model from the markup-view,
 * box model view, console, debugger, ... as well as select elements with the
 * pointer (pick).
 *
 * Other types of highlighter actors exist and can be accessed via the
 * InspectorActor's 'getHighlighterByType' method.
 */
class Highlighter extends _actor2.default {
  /**
   * Display the box model highlighting on a given NodeActor.
   * There is only one instance of the box model highlighter, so calling this
   * method several times won't display several highlighters, it will just move
   * the highlighter instance to these nodes.
   *
   * @param {NodeActor} node
   *        The node to be highlighted
   * @param {Object}    region
   *        region of box model
   * @param {Boolean}   hideInfoBar
   *        true if to hide info bar
   * @param {Boolean}   hideGuides
   *        true if to hide guides
   * @param {Boolean}   showOnly
   *        true if show only
   * @param {Boolean}   onlyRegionArea
   *        true if only region area
   * @return {Promise}  request response
   */
  showBoxModel(node, region, hideInfoBar, hideGuides, showOnly, onlyRegionArea) {
    return this.request('showBoxModel', {
      node,
      region,
      hideInfoBar,
      hideGuides,
      showOnly,
      onlyRegionArea
    });
  }
  /**
   * Hide the box model highlighting if it was shown before
   *
   * @return {Promise}  request response
   */


  hideBoxModel() {
    return this.request('hideBoxModel');
  }
  /**
   * Pick a node on click, and highlight hovered nodes in the process.
   *
   * This method doesn't respond anything interesting, however, it starts
   * mousemove, and click listeners on the content document to fire
   * events and let connected clients know when nodes are hovered over or
   * clicked.
   *
   * Once a node is picked, events will cease, and listeners will be removed.
   *
   * @return {Promise}  request response
   */


  pick() {
    return this.request('pick');
  }
  /**
   * This pick method also focuses the highlighter's target window.
   *
   * @return {Promise}  request response
   */


  pickAndFocus() {
    return this.request('pickAndFocus');
  }
  /**
   * cancel current pick
   *
   * @return {Promise}  request response
   */


  cancelPick() {
    return this.request('cancelPick');
  }

}

exports.default = Highlighter;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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