"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * An actor that returns memory usage data for its parent actor's window. A tab-scoped instance
 * of this actor will measure the memory footprint of its parent tab. A global-scoped instance
 * however, will measure the memory footprint of the chrome window referenced by the root actor.
 *
 * This actor wraps the Memory module at devtools/server/performance/memory.js
 * and provides RDP definitions.
 */
class Memory extends _actor2.default {
  constructor(client, name) {
    super(client, name);
    this.isAttached = false;
  }
  /**
   * Attach to this MemoryBridge.
   *
   * This attaches the MemoryBridge's Debugger instance so that you can start
   * recording allocations or take a census of the heap. In addition, the
   * MemoryBridge will start emitting GC events.
   *
   * @return {Promise}  request response
   */


  attach() {
    this.isAttached = true;
    return this.request('attach');
  }
  /**
   * Detach from this MemoryBridge.
   *
   * @return {Promise}  request response
   */


  detach() {
    this.isAttached = false;
    return this.request('detach');
  }
  /**
   * Gets the current MemoryBridge attach/detach state.
   *
   * @return {Promise.String}  attach/detach state
   */


  async getState() {
    const {
      state
    } = await this.request('getState');
    this.isAttached = state === 'attached';
    return state;
  }
  /**
   * Take a census of the heap. See js/src/doc/Debugger/Debugger.Memory.md for more information.
   *
   * @return {Promise}  request response
   */


  takeCensus() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    return this.request('takeCensus');
  }
  /**
   * Start recording allocation sites.
   *
   * @param {number} options.probability
   *                 The probability we sample any given allocation when recording
   *                 allocations. Must be between 0 and 1 -- defaults to 1.
   * @param {number} options.maxLogLength
   *                 The maximum number of allocation events to keep in the
   *                 log. If new allocs occur while at capacity, oldest
   *                 allocations are lost. Must fit in a 32 bit signed integer.
   * @param {number} options.drainAllocationsTimeout
   *                 A number in milliseconds of how often, at least, an `allocation`
   *                 event gets emitted (and drained), and also emits and drains on every
   *                 GC event, resetting the timer.
   * @return {Promise}  request response
   */


  async startRecordingAllocations(options) {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    const {
      value
    } = await this.request('startRecordingAllocations', {
      options
    });
    return value;
  }
  /**
   * Stop recording allocation sites.
   *
   * @return {Promise}  request response
   */


  async stopRecordingAllocations() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    const {
      value
    } = await this.request('stopRecordingAllocations');
    return value;
  }
  /**
   * Return settings used in `startRecordingAllocations` for `probability` and `maxLogLength`.
   * Currently only uses in tests.
   *
   * @return {Promise.Object}  allocation settings
   */


  async getAllocationsSettings() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    const {
      options
    } = await this.request('getAllocationsSettings');
    return options;
  }
  /**
   * Get a list of the most recent allocations since the last time we got
   * allocations, as well as a summary of all allocations since we've been
   * recording.
   *
   * @returns {Object} An object of the form:
   *
   *   {
   *     allocations: [<index into "frames" below>, ...],
   *     allocationsTimestamps: [
   *       <timestamp for allocations[0]>,
   *       <timestamp for allocations[1]>,
   *       ...
   *     ],
   *     allocationSizes: [
   *       <bytesize for allocations[0]>,
   *       <bytesize for allocations[1]>,
   *       ...
   *     ],
   *     frames: [
   *       {
   *         line: <line number for this frame>,
   *         column: <column number for this frame>,
   *         source: <filename string for this frame>,
   *         functionDisplayName:
   *           <this frame's inferred function name function or null>,
   *         parent: <index into "frames">
   *       },
   *       ...
   *     ],
   *   }
   *
   * The timestamps' unit is microseconds since the epoch.
   *
   * Subsequent `getAllocations` request within the same recording and
   * tab navigation will always place the same stack frames at the same
   * indices as previous `getAllocations` requests in the same
   * recording. In other words, it is safe to use the index as a
   * unique, persistent id for its frame.
   *
   * Additionally, the root node (null) is always at index 0.
   *
   * We use the indices into the "frames" array to avoid repeating the
   * description of duplicate stack frames both when listing
   * allocations, and when many stacks share the same tail of older
   * frames. There shouldn't be any duplicates in the "frames" array,
   * as that would defeat the purpose of this compression trick.
   *
   * In the future, we might want to split out a frame's "source" and
   * "functionDisplayName" properties out the same way we have split
   * frames out with the "frames" array. While this would further
   * compress the size of the response packet, it would increase CPU
   * usage to build the packet, and it should, of course, be guided by
   * profiling and done only when necessary.
   */


  getAllocations() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    return this.request('getAllocations');
  }
  /**
   * Force a browser-wide GC.
   *
   * @return {Promise.Object}  allocation settings
   */


  forceGarbageCollection() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    return this.request('forceGarbageCollection');
  }
  /**
   * Force an XPCOM cycle collection. For more information on XPCOM cycle
   * collection, see https://developer.mozilla.org/en-US/docs/Interfacing_with_the_XPCOM_cycle_collector#What_the_cycle_collector_does
   *
   * @return {Promise.Object}  request response
   */


  forceCycleCollection() {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    return this.request('forceCycleCollection');
  }
  /**
   * A method that returns a detailed breakdown of the memory consumption of the
   * associated window.
   *
   * @return {Object}  memory consumption
   */


  measure() {
    return this.request('measure');
  }
  /**
   * Save a heap snapshot scoped to the current debuggees' portion of the heap
   * graph.
   *
   * @param {Object|null} boundaries
   * @return {String}     The snapshot id.
   */


  async saveHeapSnapshot(boundaries) {
    if (!this.isAttached) {
      throw new Error('You need to be attached to the tab');
    }

    const {
      snapshotId
    } = await this.request('saveHeapSnapshot', {
      boundaries
    });
    return snapshotId;
  }

}

exports.default = Memory;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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