"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

var _highlighter = require("../models/highlighter");

var _highlighter2 = _interopRequireDefault(_highlighter);

var _pagestyle = require("../models/pagestyle");

var _pagestyle2 = _interopRequireDefault(_pagestyle);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Server side of the inspector actor, which is used to create
 * inspector-related actors, including the walker.
 */
class Inspector extends _actor2.default {
  /**
   * Get walker
   *
   * @param  {Object}  options  Walker options
   * @return {Promise.Object}   Walker
   */
  async getWalker(options) {
    const {
      walker
    } = await this.request('getWalker', {
      options
    });
    return walker;
  }
  /**
   * Get page style actor.
   *
   * @return {Promise.Pagestyle}  Pagestyle actor
   */


  async getPageStyle() {
    const {
      pageStyle
    } = await this.request('getPageStyle');
    return new _pagestyle2.default(this.client, pageStyle);
  }
  /**
   * The most used highlighter actor is the HighlighterActor which can be conveniently
   * retrieved by this method. The same instance will always be returned by this method
   * when called several times. The highlighter actor returned here is used to highlighter
   * elements's box-models from the markup-view, box model, console, debugger, ... as well
   * as select elements with the pointer (pick).
   *
   * @param  {Boolean}  autohide    Optionally autohide the highlighter after an element has been picked
   * @return {Promise.Highlighter}  instance of highlighter
   */


  async getHighlighter(autohide) {
    const {
      highligter
    } = this.request('getHighlighter', {
      autohide
    });
    return new _highlighter2.default(this.client, highligter);
  }
  /**
   * If consumers need to display several highlighters at the same time or different types of
   * highlighters, then this method should be used, passing the type name of the highlighter
   * needed as argument. A new instance will be created everytime the method is called, so it's
   * up to the consumer to release it when it is not needed anymore
   *
   * @param  {String}  typeName     The type of highlighter to create
   * @return {Promise.Highlighter}  The highlighter actor instance or null if the typeName passed
   *                                doesn't match any available highlighter
   */


  async getHighlighterByType(typeName) {
    const {
      highligter
    } = this.request('getHighlighterByType', {
      typeName
    });
    return new _highlighter2.default(this.client, highligter);
  }
  /**
   * Get the node's image data if any (for canvas and img nodes). Returns an imageData object with
   * the actual data being a LongStringActor and a size json object. The image data is transmitted
   * as a base64 encoded png data-uri. The method rejects if the node isn't an image or if the
   * image is missing.
   *
   * Accepts a maxDim request parameter to resize images that are larger. This is important as the
   * resizing occurs server-side so that image-data being transfered in the longstring back to the
   * client will be that much smaller.
   *
   * @param  {String} url        image url
   * @param  {Number} maxDim     resizing parameter
   * @return {Promise.<Object>}  image data
   */


  getImageDataFromURL(url, maxDim) {
    return this.request('getImageDataFromURL', {
      url,
      maxDim
    });
  }
  /**
   * Resolve a URL to its absolute form, in the scope of a given content window.
   *
   * @param  {String}    url   url to be resolved
   * @param  {NodeActor} node  If provided, the owner window of this node will be used to resolve
   *                           the URL. Otherwise, the top-level content window will be used instead.
   * @return {Promise.String}  resolved url
   */


  async resolveRelativeURL(url, node) {
    const {
      value
    } = await this.request('resolveRelativeURL', {
      url,
      node
    });
    return value;
  }
  /**
   * Pick a color from the page using the eye-dropper. This method doesn't return anything
   * but will cause events to be sent to the front when a color is picked or when the user
   * cancels the picker.
   *
   * @param  {Object} options  color picker options
   * @return {Promise}         request response
   */


  pickColorFromPage(options) {
    return this.request('pickColorFromPage', {
      options
    });
  }
  /**
   * After the pickColorFromPage method is called, the only way to dismiss the eye-dropper
   * highlighter is for the user to click in the page and select a color. If you need to
   * dismiss the eye-dropper programatically instead, use this method.
   *
   * @return {Promise}  request response
   */


  cancelPickColorFromPage() {
    return this.request('cancelPickColorFromPage');
  }
  /**
   * Check if the current document supports highlighters using a canvasFrame anonymous
   * content container (ie all highlighters except the SimpleOutlineHighlighter).
   * It is impossible to detect the feature programmatically as some document types simply
   * don't render the canvasFrame without throwing any error.
   *
   * @return {Promise}  request response
   */


  async supportsHighlighters() {
    const {
      value
    } = await this.request('supportsHighlighters');
    return value;
  }

}

exports.default = Inspector;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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