"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * This actor overrides various browser features to simulate different environments to
 * test how pages perform under various conditions.
 *
 * The design below, which saves the previous value of each property before setting, is
 * needed because it's possible to have multiple copies of this actor for a single page.
 * When some instance of this actor changes a property, we want it to be able to restore
 * that property to the way it was found before the change.
 *
 * A subtle aspect of the code below is that all get* methods must return non-undefined
 * values, so that the absence of a previous value can be distinguished from the value for
 * "no override" for each of the properties.
 */
class Emulation extends _actor2.default {
  /**
   * overwrite DPPX value
   *
   * @param  {Nimber}  dppx     dppx value
   * @return {Promise.Boolean}  true if value has changed
   */
  async setDPPXOverride(dppx) {
    const {
      valueChanged
    } = await this.request('setDPPXOverride', {
      dppx
    });
    return valueChanged;
  }
  /**
   * get current DPPX value
   *
   * @return {Promise.Number}  current dppx value
   */


  async getDPPXOverride() {
    const {
      dppx
    } = await this.request('getDPPXOverride');
    return dppx;
  }
  /**
   * clear DPPX value
   *
   * @return {Promise.Boolean}  true if value has changed
   */


  async clearDPPXOverride() {
    const {
      valueChanged
    } = await this.request('clearDPPXOverride');
    return valueChanged;
  }
  /**
   * Transform the RDP format into the internal format and then set network throttling.
   *
   * @param  {Number}  downloadThroughput  throughput in byte/s
   * @param  {Number}  uploadThroughput    throughput in byte/s
   * @param  {Number}  latency             latency time in ms
   * @return {Promise.Boolean}             true if value has changed
   */


  async setNetworkThrottling(downloadThroughput, uploadThroughput, latency) {
    const {
      valueChanged
    } = await this.request('setNetworkThrottling', {
      options: {
        downloadThroughput,
        uploadThroughput,
        latency
      }
    });
    return valueChanged;
  }
  /**
   * Get network throttling and then transform the internal format into the RDP format.
   *
   * @return {Promise.<Object>}  state of current throttle
   */


  async getNetworkThrottling() {
    const {
      state
    } = await this.request('getNetworkThrottling');
    return state;
  }
  /**
   * clear network throttling
   *
   * @return {Promise.Boolean}   true if value has changed
   */


  async clearNetworkThrottling() {
    const {
      valueChanged
    } = await this.request('clearNetworkThrottling');
    return valueChanged;
  }
  /**
   * overwrite touch events
   *
   * @param  {Boolean}  flag    true if overwrite is enabled
   * @return {Promise.Boolean}  true if value has changed
   */


  async setTouchEventsOverride(flag) {
    const {
      valueChanged
    } = await this.request('setTouchEventsOverride', {
      flag
    });
    return valueChanged;
  }
  /**
   * check if touch event overwrite is enabled
   *
   * @return {Promise.Boolean}  true if enabled
   */


  async getTouchEventsOverride() {
    const {
      flag
    } = await this.request('getTouchEventsOverride');
    return flag;
  }
  /**
   * clear state of touch event overwrite
   *
   * @return {Promise.Boolean}  true if enabled
   */


  async clearTouchEventsOverride() {
    const {
      valueChanged
    } = await this.request('clearTouchEventsOverride');
    return valueChanged;
  }
  /**
   * Overwrite user agent
   *
   * @param  {String}  userAgent  new user agent
   * @return {Promise.Boolean}    true if value has changed
   */


  async setUserAgentOverride(userAgent) {
    const {
      valueChanged
    } = await this.request('setUserAgentOverride', {
      flag: userAgent
    });
    return valueChanged;
  }
  /**
   * Get current user agent overwrite
   *
   * @return {Promise.String}  current user agent
   */


  async getUserAgentOverride() {
    const {
      userAgent
    } = await this.request('getUserAgentOverride');
    return userAgent;
  }
  /**
   * Clear user agent overwrite
   *
   * @return {Promise.Boolean}    true if value has changed
   */


  async clearUserAgentOverride() {
    const {
      valueChanged
    } = await this.request('clearUserAgentOverride');
    return valueChanged;
  }

}

exports.default = Emulation;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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