"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * CSSUsage manages the collection of CSS usage data.
 * The core of a CSSUsage is a JSON-able data structure called _knownRules
 * which looks like this:
 * This records the CSSStyleRules and their usage.
 * The format is:
 *     Map({
 *       <CSS-URL>|<START-LINE>|<START-COLUMN>: {
 *         selectorText: <CSSStyleRule.selectorText>,
 *         test: <simplify(CSSStyleRule.selectorText)>,
 *         cssText: <CSSStyleRule.cssText>,
 *         isUsed: <TRUE|FALSE>,
 *         presentOn: Set([ <HTML-URL>, ... ]),
 *         preLoadOn: Set([ <HTML-URL>, ... ]),
 *         isError: <TRUE|FALSE>,
 *       }
 *     })
 *
 * For example:
 *     this._knownRules = Map({
 *       "http://eg.com/styles1.css|15|0": {
 *         selectorText: "p.quote:hover",
 *         test: "p.quote",
 *         cssText: "p.quote { color: red; }",
 *         isUsed: true,
 *         presentOn: Set([ "http://eg.com/page1.html", ... ]),
 *         preLoadOn: Set([ "http://eg.com/page1.html" ]),
 *         isError: false,
 *       }, ...
 *     });
 */
class CSSUsage extends _actor2.default {
  /**
   * Begin recording usage data
   *
   * @param  {Boolean} [url=false] It's best if we start by reloading the current page
   *                               because that starts the test at a known point, but there could be reasons
   *                               why we don't want to do that (e.g. the page contains state that will be
   *                               lost across a reload)
   * @return {Promise.<Object>}    request response
   */
  start(url = false) {
    return this.request('start', {
      url
    });
  }
  /**
   * Cease recording usage data
   *
   * @return {Promise.<Object>}    request response
   */


  stop() {
    return this.request('stop');
  }
  /**
   * Start/stop recording usage data depending on what we're currently doing.
   *
   * @return {Promise.<Object>}    request response
   */


  toggle() {
    return this.request('toggle');
  }
  /**
   * Running start() quickly followed by stop() does a bunch of unnecessary work, so this cuts all that out
   *
   * @return {Promise.<Object>}    request response
   */


  oneshot() {
    return this.request('oneshot');
  }
  /**
   * Returns a JSONable structure designed to help marking up the style editor,
   * which describes the CSS selector usage.
   * Example:
   * [
   *     {
   *         selectorText: "p#content",
   *         usage: "unused|used",
   *         start: { line: 3, column: 0 },
   *     },
   *     ...
   * ]
   *
   * @param  {String}  url        url of page you want to audit
   * @return {Promise.<Object>}   request response
   */


  async createEditorReport(url) {
    const {
      reports
    } = await this.request('createEditorReport', {
      url
    });
    return reports;
  }
  /**
   * Compute the stylesheet URL and delegate the report creation to createEditorReport.
   * See createEditorReport documentation.
   *
   * @param  {String}  url        url of page you want to audit
   * @return {Promise.<Object>}   request response
   */


  async createEditorReportForSheet(url) {
    const {
      reports
    } = await this.request('createEditorReportForSheet', {
      url
    });
    return reports;
  }
  /**
   * Returns a JSONable structure designed for the page report which shows
   * the recommended changes to a page.
   *
   * "preload" means that a rule is used before the load event happens, which
   * means that the page could by optimized by placing it in a <style> element
   * at the top of the page, moving the <link> elements to the bottom.
   *
   * Example:
   *   {
   *     preload: [
   *       {
   *         url: "http://example.org/page1.html",
   *         shortUrl: "page1.html",
   *         rules: [
   *           {
   *             url: "http://example.org/style1.css",
   *             shortUrl: "style1.css",
   *             start: { line: 3, column: 4 },
   *             selectorText: "p#content",
   *             formattedCssText: "p#content {\n  color: red;\n }\n"
   *          },
   *          ...
   *         ]
   *       }
   *     ],
   *     unused: [
   *       {
   *         url: "http://example.org/style1.css",
   *         shortUrl: "style1.css",
   *         rules: [ ... ]
   *       }
   *     ]
   *   }
   *
   * @return {Promise.<Object>}   request response
   */


  createPageReport() {
    return this.request('createPageReport');
  }

}

exports.default = CSSUsage;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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