"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasBinary = exports.isBinary = void 0;
const withNativeBuffer = typeof Buffer === 'function' && typeof Buffer.isBuffer === 'function';
const withNativeArrayBuffer = typeof ArrayBuffer === "function";
const isView = (obj) => {
    return typeof ArrayBuffer.isView === "function"
        ? ArrayBuffer.isView(obj)
        : obj.buffer instanceof ArrayBuffer;
};
const toString = Object.prototype.toString;
const withNativeBlob = typeof Blob === "function" ||
    (typeof Blob !== "undefined" &&
        toString.call(Blob) === "[object BlobConstructor]");
const withNativeFile = typeof File === "function" ||
    (typeof File !== "undefined" &&
        toString.call(File) === "[object FileConstructor]");
/**
 * Returns true if obj is an ArrayBuffer.
 * This extra check is made because the "instanceof ArrayBuffer" check does not work
 * across different execution contexts.
 * @private
 */
function isArrayBuffer(obj) {
    return typeof obj === 'object' && obj !== null && toString.call(obj) === '[object ArrayBuffer]';
}
/**
 * Returns true if obj is a Buffer, an ArrayBuffer, a Blob or a File.
 *
 * @private
 */
function isBinary(obj) {
    return ((withNativeArrayBuffer && (obj instanceof ArrayBuffer || isArrayBuffer(obj) || isView(obj))) ||
        (withNativeBlob && obj instanceof Blob) ||
        (withNativeFile && obj instanceof File) ||
        (withNativeBuffer && Buffer.isBuffer(obj)));
}
exports.isBinary = isBinary;
function hasBinary(obj, known = [], toJSON = false) {
    if (!obj || typeof obj !== "object") {
        return false;
    }
    if (known.includes(obj)) {
        return false
    }
    known.push(obj)
    if (Array.isArray(obj)) {
        for (let i = 0, l = obj.length; i < l; i++) {
            if (hasBinary(obj[i], known)) {
                return true;
            }
        }
        return false;
    }
    if (isBinary(obj)) {
        return true;
    }
    if (obj.toJSON &&
        typeof obj.toJSON === "function" &&
        arguments.length === 2) {
        return hasBinary(obj.toJSON(), known, true);
    }
    for (const key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key) && hasBinary(obj[key], known)) {
            return true;
        }
    }
    return false;
}
exports.hasBinary = hasBinary;
