"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

var _originalsource = require("./originalsource");

var _originalsource2 = _interopRequireDefault(_originalsource);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * A StyleSheetActor represents a stylesheet on the server.
 */
class StyleSheet extends _actor2.default {
  /**
   * Toggle the disabled property of the style sheet
   *
   * @return {Promise.Object}  'disabled' - the disabled state after toggling.
   */
  async toggleDisabled() {
    const {
      disabled
    } = await this.request('toggleDisabled');
    return disabled;
  }
  /**
   * Protocol method to get the text of this stylesheet.
   *
   * @return {Promise.String}  text of stylesheet
   */


  async getText() {
    const {
      text
    } = await this.request('getText');
    return text;
  }
  /**
   * Protocol method to get the original source (actors) for this
   * stylesheet if it has uses source maps.
   *
   * @return {Promise.OriginalSource[]}  list of original sources of this stylesheet
   */


  async getOriginalSources() {
    const {
      originalSources
    } = await this.request('getOriginalSources');
    return originalSources.map(originalsource => new _originalsource2.default(this.client, originalsource));
  }
  /**
   * Protocol method that gets the location in the original source of a
   * line, column pair in this stylesheet, if its source mapped, otherwise
   * a promise of the same location.
   *
   * @return {Promise.Object} object with "source", "line" and "column" property
   */


  async getOriginalLocation(_line, _column) {
    const {
      source,
      line,
      column
    } = await this.request('getOriginalLocation', {
      line: _line,
      column: _column
    });
    return {
      source,
      line,
      column
    };
  }
  /**
   * Protocol method to get the media rules for the stylesheet.
   *
   * @return {Promise.MediaRuleActors[]}  list of media rules actors
   */


  async getMediaRules() {
    const {
      mediaRules
    } = await this.request('getMediaRules');
    return mediaRules;
  }
  /**
   * Update the style sheet in place with new text.
   *
   * @param  {object} request
   *         'text' - new text
   *         'transition' - whether to do CSS transition for change.
   * @return {Promise.Object}  request response
   */


  update(text, transition) {
    return this.request('update', {
      text,
      transition
    });
  }

}

exports.default = StyleSheet;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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