"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _os = require("os");

var _os2 = _interopRequireDefault(_os);

var _fsExtra = require("fs-extra");

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require("path");

var _path2 = _interopRequireDefault(_path);

var _child_process = require("child_process");

var _getPort = require("get-port");

var _getPort2 = _interopRequireDefault(_getPort);

var _tcpPortUsed = require("tcp-port-used");

var _browser = require("./browser");

var _browser2 = _interopRequireDefault(_browser);

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const CONNECTION_TIMEOUT = 10000;
const CONNECTION_INTERVAL = 250;

const FIREFOX_PROFILE_PATH = _path2.default.join(_os2.default.tmpdir(), 'firefox_dev_profile-');

class Launcher {
  static async launch(options = {}) {
    const host = '127.0.0.1';
    const listeners = [];
    const debuggerPort = await (0, _getPort2.default)({
      port: options.port || 9222
    });
    /**
     * create custom profile
     */

    const profileDir = _fsExtra2.default.mkdtempSync(FIREFOX_PROFILE_PATH);

    _fsExtra2.default.copySync(_path2.default.resolve(__dirname, 'config', 'profile', 'prefs.js'), _path2.default.resolve(profileDir, 'prefs.js'));

    const firefoxExecuteable = options.bin || Launcher.getFirefoxBin(); // Enable -P option over -profile

    if (!options.args || !options.args.includes('-P')) {
      options.args = (options.args || []).concat('-profile', options.profile || profileDir);
    }

    const firefoxArguments = options.args.concat('-start-debugger-server', debuggerPort, '-url', options.url || 'https://google.com', '-override', _path2.default.resolve(__dirname, 'config', 'override.ini'));
    const firefoxProcess = (0, _child_process.spawn)(firefoxExecuteable, firefoxArguments, {});

    function killAndCleanup() {
      (0, _utils.removeEventListeners)(listeners);
      firefoxProcess.kill('SIGKILL');
    }

    listeners.push((0, _utils.addEventListener)(process, 'exit', killAndCleanup));
    listeners.push((0, _utils.addEventListener)(firefoxProcess, 'exit', killAndCleanup));
    await Launcher.waitUntilConnected(host, debuggerPort);
    const browser = new _browser2.default(host, debuggerPort);
    let tabs = await browser.connect();
    browser.firefoxProcess = firefoxProcess;
    return Launcher.waitForTab(browser, tabs);
  }

  static getFirefoxExe() {
    // Only run these checks on win32
    if (process.platform !== 'win32') {
      return null;
    }

    const suffix = '\\Mozilla Firefox\\firefox.exe';
    const prefixes = [process.env.LOCALAPPDATA, process.env.PROGRAMFILES, process.env['PROGRAMFILES(X86)']];
    let windowsFirefoxDirectory;

    for (const prefix of prefixes) {
      try {
        windowsFirefoxDirectory = _path2.default.join(prefix, suffix);

        _fsExtra2.default.accessSync(windowsFirefoxDirectory);

        return windowsFirefoxDirectory;
      } catch (e) {}
    }

    return windowsFirefoxDirectory;
  }

  static getFirefoxBin() {
    if (process.platform === 'win32') {
      return Launcher.getFirefoxExe();
    }

    if (process.platform === 'darwin') {
      return '/Applications/Firefox.app/Contents/MacOS/firefox';
    }

    if (process.platform === 'linux') {
      return 'firefox';
    }

    throw new Error(`Couldn't find executable for platform "${process.platform}"`);
  }

  static async waitUntilConnected(host, port, timeout = CONNECTION_TIMEOUT) {
    const isConnected = await (0, _tcpPortUsed.check)(port, host);

    if (isConnected) {
      return true;
    }

    await new Promise(resolve => setTimeout(resolve, CONNECTION_INTERVAL));
    return Launcher.waitUntilConnected(host, port, timeout - CONNECTION_INTERVAL);
  }

  static async waitForTab(browser, tabs) {
    if (!tabs) {
      tabs = await browser.listTabs();
    }

    if (tabs.length > 0) {
      return {
        browser,
        tab: tabs[0]
      };
    }

    await new Promise(resolve => setTimeout(resolve, CONNECTION_INTERVAL));
    return Launcher.waitForTab(browser);
  }

}

exports.default = Launcher;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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