"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Main interface for interacting with nsIProfiler
 */
class Profiler extends _actor2.default {
  /**
   * Starts the nsIProfiler module. Doing so will discard any samples
   * that might have been accumulated so far.
   *
   * @param {Number} entries               number of entries
   * @param {Number} interval              recording interval
   * @param {Array<String>} features       list of features to include
   * @param {Array<String>} threadFilters  list of thread filters
   *
   * @return {Promise.Object}              request response
   */
  startProfiler(entries, interval, features, threadFilters) {
    return this.request('startProfiler', {
      entries,
      interval,
      features,
      threadFilters
    });
  }
  /**
   * Attempts to stop the nsIProfiler module.
   * @return {Promise.Object}  request response
   */


  stopProfiler() {
    return this.request('stopProfiler');
  }
  /**
   * Returns all the samples accumulated since the profiler was started,
   * along with the current time. The data has the following format:
   * {
   *   libs: string,
   *   meta: {
   *     interval: number,
   *     platform: string,
   *     ...
   *   },
   *   threads: [{
   *     samples: [{
   *       frames: [{
   *         line: number,
   *         location: string,
   *         category: number
   *       } ... ],
   *       name: string
   *       responsiveness: number
   *       time: number
   *     } ... ]
   *   } ... ]
   * }
   *
   *
   * @param {Number} startTime
   *        Since the circular buffer will only grow as long as the profiler lives,
   *        the buffer can contain unwanted samples. Pass in a `startTime` to only
   *        retrieve samples that took place after the `startTime`, with 0 being
   *        when the profiler just started.
   * @param {Boolean} stringify
   *        Whether or not the returned profile object should be a string or not to
   *        save JSON parse/stringify cycle if emitting over RDP.
   */


  getProfile(startTime, stringify) {
    return this.request('getProfile', {
      startTime,
      stringify
    });
  }
  /**
   * Returns an array of feature strings, describing the profiler features
   * that are available on this platform. Can be called while the profiler
   * is stopped.
   *
   * @return {Promise.String[]}  list of feature strings
   */


  async getFeatures() {
    const {
      features
    } = await this.request('getFeatures');
    return features;
  }
  /**
   * Returns an object with the values of the current status of the
   * circular buffer in the profiler, returning `position`, `totalSize`,
   * and the current `generation` of the buffer.
   *
   * @return {Promise.Object}  current status of the circular buffer
   */


  getBufferInfo() {
    return this.request('getBufferInfo');
  }
  /**
   * Returns the configuration used that was originally passed in to start up the
   * profiler. Used for tests, and does not account for others using nsIProfiler.
   *
   * @return {Promise.Object}  profiler configurations
   */


  getStartOptions() {
    return this.request('getStartOptions');
  }
  /**
   * Verifies whether or not the nsIProfiler module has started.
   * If already active, the current time is also returned.
   *
   * @return {Promise.Boolean}  true if nsIProfiler module has started
   */


  isActive() {
    return this.request('isActive');
  }
  /**
   * Returns an array of objects that describes the shared libraries
   * which are currently loaded into our process. Can be called while the
   * profiler is stopped.
   *
   * @return {Promise.Object[]}  list of objects that describes the shared libraries
   */


  sharedLibraries() {
    return this.request('sharedLibraries');
  }
  /**
   * Registers handlers for the following events to be emitted
   * on active Profiler instances:
   *   - "console-api-profiler"
   *   - "profiler-started"
   *   - "profiler-stopped"
   *   - "profiler-status"
   *
   * The ProfilerManager listens to all events, and individual
   * consumers filter which events they are interested in.
   *
   * @param  {String[]} events  events to listen to
   * @return {Promise}          request response
   */


  registerEventNotifications(events) {
    return this.request('registerEventNotifications', {
      events
    });
  }
  /**
   * Unregisters handlers for all system events.
   *
   * @param  {String[]} events  events to unregister of
   * @return {Promise}          request response
   */


  unregisterEventNotifications(events) {
    return this.request('unregisterEventNotifications', {
      events
    });
  }
  /**
   * Updates the frequency that the "profiler-status" event is emitted
   * during recording.
   *
   * @param {Number} interval  interval of recording
   */


  setProfilerStatusInterval(interval) {
    return this.request('setProfilerStatusInterval', {
      interval
    });
  }

}

exports.default = Profiler;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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