"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("../actor");

var _actor2 = _interopRequireDefault(_actor);

var _utils = require("../utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Console extends _actor2.default {
  constructor(client, name) {
    super(client, name);
    this.consoleApiCall = null;
    this.pageErrorCall = null;
    this.on('consoleAPICall', event => {
      if (this.consoleApiCall) {
        this.consoleApiCall(event);
      }
    });
    this.on('pageError', event => {
      if (this.pageErrorCall) {
        this.pageErrorCall(event);
      }
    });
    this.on('evaluationResult', event => {
      this.emit(`evaluationResult-${event.resultID}`, event);
    });
  }
  /**
   * Start the given Web Console listeners.
   *
   * @param  {String[]} listeners  listeners to listen to (default: 'PageError', 'ConsoleAPI')
   * @return {Promise.<Object>}    request response
   */


  startListeners(listeners = Console.listeners) {
    this.isEnabled = true;
    return this.request('startListeners', {
      listeners
    });
  }
  /**
   * Stop the given Web Console listeners.
   *
   * @param  {String[]} listeners  listeners to stop listen to (default: 'PageError', 'ConsoleAPI')
   * @return {Promise.<Object>}    request response
   */


  stopListeners(listeners = Console.listeners) {
    this.isEnabled = false;
    return this.request('stopListeners', {
      listeners
    });
  }
  /**
   * Retrieve the cached messages from the server.
   *
   * @param  {String[]}  messageTypes  type of cached message to get (default: 'PageError', 'ConsoleAPI')
   * @return {Promise.<Object>[]}      list of console messages
   */


  async getCachedMessages(messageTypes = Console.listeners) {
    if (!this.isEnabled) {
      throw new Error('Console is not enabled');
    }

    const resp = await this.request('getCachedMessages', {
      messageTypes
    });
    return resp.messages;
  }
  /**
   * Clear the cache of messages (console API calls only).
   *
   * @return {Promise.<Object>}  request response
   */


  clearMessagesCache() {
    return this.request('clearMessagesCache');
  }
  /**
   * Evaluate a JavaScript expression.
   * Starting from FF72, this method becomes deprecated
   * Please use evaluateJSAsync instead
   *
   * @param  {String|Function}  script  js code to evaluate
   * @param  {Object}           args    arguments to pass to the function
   * @return {Promise.<Object, Error>}  result of the js function or an exception if script fails
   */


  async evaluateJS(script, ...args) {
    const argsTransformed = (0, _utils.transformEvaluateArgs)(args);
    const text = (0, _utils.transformEvaluateScript)(script, argsTransformed);
    const {
      result,
      exception,
      exceptionMessage
    } = await this.request('evaluateJS', {
      text
    });

    if (exception) {
      throw new Error(exceptionMessage);
    }

    return result;
  }
  /**
   * Evaluate a JavaScript expression async.
   *
   * @param  {String|Function}  script  js code to evaluate
   * @param  {Object}           args    arguments to pass to the function
   * @return {Promise.<Object, Error>}  result of the js function or an exception if script fails
   */


  async evaluateJSAsync(script, ...args) {
    const argsTransformed = (0, _utils.transformEvaluateArgs)(args);
    const text = (0, _utils.transformEvaluateScript)(script, argsTransformed);
    const {
      resultID
    } = await this.request('evaluateJSAsync', {
      text
    });
    return new Promise((resolve, reject) => {
      this.on(`evaluationResult-${resultID}`, message => {
        if (message.exception) {
          reject(new Error(message.exceptionMessage));
        }

        resolve(message.result);
      });
    });
  }
  /**
   * Get Web Console-related preferences on the server.
   *
   * @param  {String[]} preferences  An array with the preferences you want to retrieve.
   * @return {Promise.<Object>[]}    List of preferences
   */


  getPreferences(preferences) {
    return this.request('getPreferences', {
      preferences
    });
  }
  /**
   * Set Web Console-related preferences on the server.
   *
   * @param {Object} preferences  An object with the preferences you want to change.
   * @return {Promise.<Object>}   request response
   */


  setPreferences(preferences) {
    return this.request('setPreferences', {
      preferences
    });
  }
  /**
   * Autocomplete a JavaScript expression.
   *
   * @param  {String} text      The code you want to autocomplete.
   * @param  {Number} cursor    Cursor location inside the string. Index starts from 0.
   * @return {Promise.<Object>} request response
   */


  autocomplete(text, cursor) {
    return this.request('autocomplete', {
      text,
      cursor
    });
  }
  /**
   * Inspect the properties of an object.
   *
   * @return {Promise.<Object>} request response
   */


  inspectObjectProperties() {
    return this.request('inspectProperties');
  }
  /**
   * Set a callback function to be invoked at ConsoleApiCall event
   *
   * @param {Function} callback
   */


  onConsoleApiCall(callback) {
    this.consoleApiCall = callback;
  }
  /**
   * Set a callback function to be invoked at PageError event
   *
   * @param {Function} callback
   */


  onPageError(callback) {
    this.pageErrorCall = callback;
  }

}

exports.default = Console;

_defineProperty(Console, "listeners", ['PageError', 'ConsoleAPI']);

module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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