"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _actor = require("./actor");

var _actor2 = _interopRequireDefault(_actor);

var _client = require("./client");

var _client2 = _interopRequireDefault(_client);

var _tab = require("./tab");

var _tab2 = _interopRequireDefault(_tab);

var _os = require("os");

var _os2 = _interopRequireDefault(_os);

var _child_process = require("child_process");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Browser extends _actor2.default {
  constructor(host, port) {
    const client = new _client2.default(host, port);
    client.on('error', error => this.emit('error', error));
    client.on('end', () => this.emit('end'));
    client.on('timeout', () => this.emit('timeout'));
    super(client, 'root');
    this.cachedTabs = new Map();
  }

  get preference() {
    return this._get('preference');
  }

  get actorRegistry() {
    return this._get('actorRegistry');
  }

  get addons() {
    return this._get('addons');
  }

  get device() {
    return this._get('device');
  }

  get heapSnapshotFile() {
    return this._get('heapSnapshotFile');
  }

  async connect() {
    await this.client.connect();
    this.client.expectReply(this.name, ({
      traits
    }) => {
      this.traits = traits;
    });
    this.tabs = await this.listTabs();
    return this.tabs;
  }

  disconnect() {
    this.client.disconnect();
  }

  close() {
    this.disconnect();
    /**
     * only shut down browser if started via launcher
     */

    if (!this.firefoxProcess) {
      return console.error('Can\'t close the browser because client was attached to an' + 'already opened Firefox instance');
    }

    this.firefoxProcess.kill();

    if (!this.firefoxProcess.killed) {
      if (_os2.default.platform() === 'win32') {
        (0, _child_process.spawn)('taskkill', ['/f', '/IM', 'firefox.exe', '/t']);
      } else if (_os2.default.platform() === 'darwin' || _os2.default.platform() === 'linux') {
        (0, _child_process.spawn)('killall', ['-q', '-I', 'firefox']);
      }
    }
  }

  async listTabs() {
    let listTabsResponse = await this.request('listTabs');

    if (!listTabsResponse.tabs) {
      listTabsResponse = await this.request('listTabs');
    }

    this.setActors(listTabsResponse);
    const tabList = await Promise.all(listTabsResponse.tabs.map(async tab => {
      if (this.cachedTabs.has(tab.actor)) {
        return this.cachedTabs.get(tab.actor);
      }

      let newTab = new _tab2.default(this.client, tab.actor, tab); // Firefox 75: getTarget

      if (!tab.consoleActor) {
        await newTab.getTarget();
      }

      this.cachedTabs.set(tab.actor, newTab);
      return newTab;
    }));

    this._cleanCache(listTabsResponse.tabs.map(tab => tab.actor));

    return tabList;
  }

  _cleanCache(activeTabs) {
    Array.from(this.cachedTabs.keys()).forEach(tabName => {
      if (!activeTabs.includes(tabName)) {
        this.cachedTabs.delete(tabName);
      }
    });
  }

}

exports.default = Browser;
module.exports = exports["default"];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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