"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectActions = void 0;
const tslib_1 = require("tslib");
const types_1 = require("../../../types");
const execa_1 = tslib_1.__importDefault(require("execa"));
const path_1 = tslib_1.__importDefault(require("path"));
const assert_1 = tslib_1.__importDefault(require("assert"));
const codegen_1 = require("../codegen");
const templates_1 = tslib_1.__importDefault(require("../codegen/templates"));
const util_1 = require("../util");
const errors_1 = require("../../../errors");
const config_1 = require("../../../config");
class ProjectActions {
    constructor(ctx) {
        this.ctx = ctx;
    }
    get api() {
        return this.ctx._apis.projectApi;
    }
    async clearCurrentProject() {
        this.ctx.update((d) => {
            d.activeBrowser = null;
            d.currentProject = null;
            d.diagnostics = {
                error: null,
                warnings: [],
            };
            d.currentTestingType = null;
            d.forceReconfigureProject = null;
            d.scaffoldedFiles = null;
            d.app.browserStatus = 'closed';
        });
        await this.ctx.lifecycleManager.clearCurrentProject();
        (0, config_1.resetIssuedWarnings)();
        await this.api.closeActiveProject();
    }
    get projects() {
        return this.ctx.projectsList;
    }
    set projects(projects) {
        this.ctx.coreData.app.projects = projects;
    }
    openDirectoryInIDE(projectPath) {
        this.ctx.debug(`opening ${projectPath} in ${this.ctx.coreData.localSettings.preferences.preferredEditorBinary}`);
        if (!this.ctx.coreData.localSettings.preferences.preferredEditorBinary) {
            return;
        }
        if (this.ctx.coreData.localSettings.preferences.preferredEditorBinary === 'computer') {
            this.ctx.actions.electron.showItemInFolder(projectPath);
        }
        (0, execa_1.default)(this.ctx.coreData.localSettings.preferences.preferredEditorBinary, [projectPath]);
    }
    setAndLoadCurrentTestingType(type) {
        this.ctx.lifecycleManager.setAndLoadCurrentTestingType(type);
    }
    async setCurrentProject(projectRoot) {
        await this.updateProjectList(() => this.api.insertProjectToCache(projectRoot));
        await this.clearCurrentProject();
        await this.ctx.lifecycleManager.setCurrentProject(projectRoot);
    }
    // Temporary: remove after other refactor lands
    async setCurrentProjectAndTestingTypeForTestSetup(projectRoot) {
        await this.ctx.lifecycleManager.clearCurrentProject();
        await this.ctx.lifecycleManager.setCurrentProject(projectRoot);
        this.ctx.lifecycleManager.setCurrentTestingType('e2e');
        // @ts-expect-error - we are setting this as a convenience for our integration tests
        this.ctx._modeOptions = {};
    }
    async loadProjects() {
        const projectRoots = await this.api.getProjectRootsFromCache();
        this.ctx.update((d) => {
            d.app.projects = [...projectRoots];
        });
        return this.projects;
    }
    async initializeActiveProject(options = {}) {
        (0, assert_1.default)(this.ctx.currentProject, 'Cannot initialize project without an active project');
        (0, assert_1.default)(this.ctx.coreData.currentTestingType, 'Cannot initialize project without choosing testingType');
        const allModeOptionsWithLatest = {
            ...this.ctx.modeOptions,
            projectRoot: this.ctx.currentProject,
            testingType: this.ctx.coreData.currentTestingType,
        };
        try {
            await this.api.closeActiveProject();
            return await this.api.openProjectCreate(allModeOptionsWithLatest, {
                ...options,
                ctx: this.ctx,
            }).finally(async () => {
                // When switching testing type, the project should be relaunched in the previously selected browser
                if (this.ctx.coreData.app.relaunchBrowser) {
                    this.ctx.project.setRelaunchBrowser(false);
                    await this.ctx.actions.project.launchProject(this.ctx.coreData.currentTestingType);
                }
            });
        }
        catch (e) {
            // TODO(tim): remove / replace with ctx.log.error
            // eslint-disable-next-line
            console.error(e);
            throw e;
        }
    }
    async updateProjectList(updater) {
        return updater().then(() => this.loadProjects());
    }
    async addProjectFromElectronNativeFolderSelect() {
        const path = await this.ctx.actions.electron.showOpenDialog();
        if (!path) {
            return;
        }
        await this.addProject({ path, open: true });
        this.ctx.emitter.toLaunchpad();
    }
    async addProject(args) {
        const projectRoot = await this.getDirectoryPath(args.path);
        if (args.open) {
            this.setCurrentProject(projectRoot).catch(this.ctx.onError);
        }
        else {
            await this.updateProjectList(() => this.api.insertProjectToCache(projectRoot));
        }
    }
    async getDirectoryPath(projectRoot) {
        try {
            const { dir, base } = path_1.default.parse(projectRoot);
            const fullPath = path_1.default.join(dir, base);
            const dirStat = await this.ctx.fs.stat(fullPath);
            if (dirStat.isDirectory()) {
                return fullPath;
            }
            return dir;
        }
        catch (exception) {
            throw Error(`Cannot add ${projectRoot} to projects as it does not exist in the file system`);
        }
    }
    async launchProject(testingType, options, specPath) {
        if (!this.ctx.currentProject) {
            return null;
        }
        testingType = testingType || this.ctx.coreData.currentTestingType;
        // It's strange to have no testingType here, but `launchProject` is called when switching testing types,
        // so it needs to short-circuit and return here.
        // TODO: Untangle this. https://cypress-io.atlassian.net/browse/UNIFY-1528
        if (!testingType)
            return;
        this.ctx.coreData.currentTestingType = testingType;
        const browser = this.ctx.coreData.activeBrowser;
        if (!browser)
            throw new Error('Missing browser in launchProject');
        let activeSpec;
        if (specPath) {
            activeSpec = specPath === types_1.RUN_ALL_SPECS_KEY ? types_1.RUN_ALL_SPECS : this.ctx.project.getCurrentSpecByAbsolute(specPath);
        }
        // launchProject expects a spec when opening browser for url navigation.
        // We give it an template spec if none is passed so as to land on home page
        const emptySpec = {
            name: '',
            absolute: '',
            relative: '',
            specType: testingType === 'e2e' ? 'integration' : 'component',
        };
        // Used for run-all-specs feature
        if (options === null || options === void 0 ? void 0 : options.shouldLaunchNewTab) {
            await this.api.resetBrowserTabsForNextTest(true);
            this.api.resetServer();
        }
        await this.api.launchProject(browser, activeSpec !== null && activeSpec !== void 0 ? activeSpec : emptySpec, options);
        return;
    }
    removeProject(projectRoot) {
        return this.updateProjectList(() => this.api.removeProjectFromCache(projectRoot));
    }
    async createConfigFile(type) {
        const project = this.ctx.currentProject;
        if (!project) {
            throw Error(`Cannot create config file without currentProject.`);
        }
        let obj = {
            e2e: {},
            component: {},
        };
        if (type) {
            obj = {
                [type]: {},
            };
        }
        await this.ctx.fs.writeFile(this.ctx.lifecycleManager.configFilePath, `module.exports = ${JSON.stringify(obj, null, 2)}`);
    }
    async setProjectIdInConfigFile(projectId) {
        return (0, util_1.insertValuesInConfigFile)(this.ctx.lifecycleManager.configFilePath, { projectId }, { get(id) {
                return Error(id);
            } });
    }
    async clearLatestProjectCache() {
        await this.api.clearLatestProjectsCache();
    }
    async clearProjectPreferencesCache(projectTitle) {
        await this.api.clearProjectPreferences(projectTitle);
    }
    async clearAllProjectPreferencesCache() {
        await this.api.clearAllProjectPreferences();
    }
    setPromptShown(slug) {
        this.api.setPromptShown(slug);
    }
    setSpecs(specs) {
        this.ctx.project.setSpecs(specs);
        this.refreshSpecs(specs);
        // only check for non-example specs when the specs change
        this.hasNonExampleSpec().then((result) => {
            this.ctx.project.setHasNonExampleSpec(result);
        })
            .catch((e) => {
            this.ctx.project.setHasNonExampleSpec(false);
            this.ctx.logTraceError(e);
        });
        if (this.ctx.coreData.currentTestingType === 'component') {
            this.api.getDevServer().updateSpecs(specs);
        }
        this.ctx.emitter.specsChange();
    }
    refreshSpecs(specs) {
        var _a;
        (_a = this.ctx.lifecycleManager.git) === null || _a === void 0 ? void 0 : _a.setSpecs(specs.map((s) => s.absolute));
    }
    setProjectPreferencesInGlobalCache(args) {
        if (!this.ctx.currentProject) {
            throw Error(`Cannot save preferences without currentProject.`);
        }
        this.api.insertProjectPreferencesToCache(this.ctx.lifecycleManager.projectTitle, args);
    }
    async setSpecsFoundBySpecPattern({ projectRoot, testingType, specPattern, configSpecPattern, excludeSpecPattern, additionalIgnorePattern }) {
        const toArray = (val) => val ? typeof val === 'string' ? [val] : val : [];
        configSpecPattern = toArray(configSpecPattern);
        specPattern = toArray(specPattern);
        excludeSpecPattern = toArray(excludeSpecPattern) || [];
        // exclude all specs matching e2e if in component testing
        additionalIgnorePattern = toArray(additionalIgnorePattern) || [];
        if (!specPattern || !configSpecPattern) {
            throw Error('could not find pattern to load specs');
        }
        const specs = await this.ctx.project.findSpecs({
            projectRoot,
            testingType,
            specPattern,
            configSpecPattern,
            excludeSpecPattern,
            additionalIgnorePattern,
        });
        this.ctx.actions.project.setSpecs(specs);
        await this.ctx.project.startSpecWatcher({
            projectRoot,
            testingType,
            specPattern,
            configSpecPattern,
            excludeSpecPattern,
            additionalIgnorePattern,
        });
    }
    setForceReconfigureProjectByTestingType({ forceReconfigureProject, testingType }) {
        const testingTypeToReconfigure = testingType !== null && testingType !== void 0 ? testingType : this.ctx.coreData.currentTestingType;
        if (!testingTypeToReconfigure) {
            return;
        }
        this.ctx.update((coreData) => {
            coreData.forceReconfigureProject = {
                ...coreData.forceReconfigureProject,
                [testingTypeToReconfigure]: forceReconfigureProject,
            };
        });
    }
    async reconfigureProject() {
        await this.ctx.actions.browser.closeBrowser();
        this.ctx.actions.wizard.resetWizard();
        await this.ctx.actions.wizard.initialize();
        this.ctx.actions.electron.refreshBrowserWindow();
        this.ctx.actions.electron.showBrowserWindow();
    }
    async hasNonExampleSpec() {
        var _a;
        const specs = (_a = this.ctx.project.specs) === null || _a === void 0 ? void 0 : _a.map((spec) => spec.relativeToCommonRoot);
        switch (this.ctx.coreData.currentTestingType) {
            case 'e2e':
                return (0, codegen_1.hasNonExampleSpec)(templates_1.default.scaffoldIntegration, specs);
            case 'component':
                return specs.length > 0;
            case null:
                return false;
            default:
                throw new Error(`Unsupported testing type ${this.ctx.coreData.currentTestingType}`);
        }
    }
    async pingBaseUrl() {
        var _a;
        const baseUrl = (_a = (await this.ctx.project.getConfig())) === null || _a === void 0 ? void 0 : _a.baseUrl;
        // Should never happen
        if (!baseUrl) {
            return;
        }
        const baseUrlWarning = this.ctx.warnings.find((e) => e.cypressError.type === 'CANNOT_CONNECT_BASE_URL_WARNING');
        if (baseUrlWarning) {
            this.ctx.actions.error.clearWarning(baseUrlWarning.id);
            this.ctx.emitter.errorWarningChange();
        }
        return this.api.isListening(baseUrl)
            .catch(() => this.ctx.onWarning((0, errors_1.getError)('CANNOT_CONNECT_BASE_URL_WARNING', baseUrl)));
    }
    async switchTestingTypesAndRelaunch(testingType) {
        const isTestingTypeConfigured = this.ctx.lifecycleManager.isTestingTypeConfigured(testingType);
        this.ctx.project.setRelaunchBrowser(isTestingTypeConfigured);
        this.setAndLoadCurrentTestingType(testingType);
        await this.reconfigureProject();
        if (testingType === 'e2e' && !isTestingTypeConfigured) {
            // E2E doesn't have a wizard, so if we have a testing type on load we just create/update their cypress.config.js.
            await this.ctx.actions.wizard.scaffoldTestingType();
        }
    }
}
exports.ProjectActions = ProjectActions;
