"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isStringOrArrayOfStrings = exports.isFullyQualifiedUrl = exports.isStringOrFalse = exports.isNumberOrFalse = exports.isArray = exports.isString = exports.isNumber = exports.isBoolean = exports.isPlainObject = exports.isValidClientCertificatesSet = exports.isOneOf = exports.isValidRetriesConfig = exports.isValidBrowserList = exports.isValidBrowser = void 0;
const tslib_1 = require("tslib");
const url_1 = require("url");
const path_1 = tslib_1.__importDefault(require("path"));
const _ = tslib_1.__importStar(require("lodash"));
const is = tslib_1.__importStar(require("check-more-types"));
const common_tags_1 = require("common-tags");
const debug_1 = tslib_1.__importDefault(require("debug"));
const types_1 = require("../../types");
const debug = (0, debug_1.default)('cypress:server:validation');
// validation functions take a key and a value and should:
//  - return true if it passes validation
//  - return a error message if it fails validation
const str = JSON.stringify;
const errMsg = (key, value, type) => {
    return {
        key,
        value,
        type,
    };
};
const _isFullyQualifiedUrl = (value) => {
    return _.isString(value) && /^https?\:\/\//.test(value);
};
const isArrayOfStrings = (value) => {
    return _.isArray(value) && _.every(value, _.isString);
};
const isFalse = (value) => {
    return value === false;
};
/**
 * Validates a single browser object.
 * @returns {string|true} Returns `true` if the object is matching browser object schema. Returns an error message if it does not.
 */
const isValidBrowser = (browser) => {
    if (!is.unemptyString(browser.name)) {
        return errMsg('name', browser, 'a non-empty string');
    }
    if (!is.oneOf(types_1.BROWSER_FAMILY)(browser.family)) {
        return errMsg('family', browser, (0, common_tags_1.commaListsOr) `either ${types_1.BROWSER_FAMILY}`);
    }
    if (!is.unemptyString(browser.displayName)) {
        return errMsg('displayName', browser, 'a non-empty string');
    }
    if (!is.unemptyString(browser.version)) {
        return errMsg('version', browser, 'a non-empty string');
    }
    if (typeof browser.path !== 'string') {
        return errMsg('path', browser, 'a string');
    }
    if (typeof browser.majorVersion !== 'string' && !(is.number(browser.majorVersion) && browser.majorVersion > 0)) {
        return errMsg('majorVersion', browser, 'a string or a positive number');
    }
    return true;
};
exports.isValidBrowser = isValidBrowser;
/**
 * Validates the list of browsers.
 */
const isValidBrowserList = (_key, browsers) => {
    debug('browsers %o', browsers);
    if (!browsers) {
        return 'Missing browsers list';
    }
    if (!Array.isArray(browsers)) {
        debug('browsers is not an array', typeof browsers);
        return 'Browsers should be an array';
    }
    if (!browsers.length) {
        return 'Expected at least one browser';
    }
    for (let k = 0; k < browsers.length; k += 1) {
        const validationResult = (0, exports.isValidBrowser)(browsers[k]);
        if (validationResult !== true) {
            validationResult.list = 'browsers';
            return validationResult;
        }
    }
    return true;
};
exports.isValidBrowserList = isValidBrowserList;
const isValidRetriesConfig = (key, value) => {
    const optionalKeys = ['runMode', 'openMode'];
    const isValidRetryValue = (val) => _.isNull(val) || (Number.isInteger(val) && val >= 0);
    const optionalKeysAreValid = (val, k) => optionalKeys.includes(k) && isValidRetryValue(val);
    if (isValidRetryValue(value)) {
        return true;
    }
    if (_.isObject(value) && _.every(value, optionalKeysAreValid)) {
        return true;
    }
    return errMsg(key, value, 'a positive number or null or an object with keys "openMode" and "runMode" with values of numbers or nulls');
};
exports.isValidRetriesConfig = isValidRetriesConfig;
/**
 * Checks if given value for a key is equal to one of the provided values.
 * @example
  ```
  validate = v.isOneOf("foo", "bar")
  validate("example", "foo") // true
  validate("example", "else") // error message string
  ```
  */
const isOneOf = (...values) => {
    return (key, value) => {
        if (values.some((v) => {
            if (typeof value === 'function') {
                return value(v);
            }
            return v === value;
        })) {
            return true;
        }
        const strings = values.map((a) => str(a)).join(', ');
        return errMsg(key, value, `one of these values: ${strings}`);
    };
};
exports.isOneOf = isOneOf;
/**
 * Validates whether the supplied set of cert information is valid
 * @returns {string|true} Returns `true` if the information set is valid. Returns an error message if it is not.
 */
// _key: string, certsForUrls: any[]): ErrResult | true {}
const isValidClientCertificatesSet = (_key, certsForUrls) => {
    debug('clientCerts: %o', certsForUrls);
    if (!Array.isArray(certsForUrls)) {
        return errMsg(`clientCertificates.certs`, certsForUrls, 'an array of certs for URLs');
    }
    let urls = [];
    for (let i = 0; i < certsForUrls.length; i++) {
        debug(`Processing clientCertificates: ${i}`);
        let certsForUrl = certsForUrls[i];
        if (!certsForUrl) {
            continue;
        }
        if (!certsForUrl.url) {
            return errMsg(`clientCertificates[${i}].url`, certsForUrl.url, 'a URL matcher');
        }
        if (certsForUrl.url !== '*') {
            try {
                let parsed = new url_1.URL(certsForUrl.url);
                if (parsed.protocol !== 'https:') {
                    return errMsg(`clientCertificates[${i}].url`, certsForUrl.url, 'an https protocol');
                }
            }
            catch (e) {
                return errMsg(`clientCertificates[${i}].url`, certsForUrl.url, 'a valid URL');
            }
        }
        if (urls.includes(certsForUrl.url)) {
            return `clientCertificates has duplicate client certificate URL: ${certsForUrl.url}`;
        }
        urls.push(certsForUrl.url);
        if (certsForUrl.ca && !Array.isArray(certsForUrl.ca)) {
            return errMsg(`clientCertificates[${i}].ca`, certsForUrl.ca, 'an array of CA filepaths');
        }
        if (!Array.isArray(certsForUrl.certs)) {
            return errMsg(`clientCertificates[${i}].certs`, certsForUrl.certs, 'an array of certs');
        }
        for (let j = 0; j < certsForUrl.certs.length; j++) {
            let certInfo = certsForUrl.certs[j];
            // Only one of PEM or PFX cert allowed
            if (certInfo.cert && certInfo.pfx) {
                return `\`clientCertificates[${i}].certs[${j}]\` has both PEM and PFX defined`;
            }
            if (!certInfo.cert && !certInfo.pfx) {
                return `\`clientCertificates[${i}].certs[${j}]\` must have either PEM or PFX defined`;
            }
            if (certInfo.pfx) {
                if (path_1.default.isAbsolute(certInfo.pfx)) {
                    return errMsg(`clientCertificates[${i}].certs[${j}].pfx`, certInfo.pfx, 'a relative filepath');
                }
            }
            if (certInfo.cert) {
                if (path_1.default.isAbsolute(certInfo.cert)) {
                    return errMsg(`clientCertificates[${i}].certs[${j}].cert`, certInfo.cert, 'a relative filepath');
                }
                if (!certInfo.key) {
                    return errMsg(`clientCertificates[${i}].certs[${j}].key`, certInfo.key, 'a key filepath');
                }
                if (path_1.default.isAbsolute(certInfo.key)) {
                    return errMsg(`clientCertificates[${i}].certs[${j}].key`, certInfo.key, 'a relative filepath');
                }
            }
        }
        if (certsForUrl.ca) {
            for (let k = 0; k < certsForUrl.ca.length; k++) {
                if (path_1.default.isAbsolute(certsForUrl.ca[k] || '')) {
                    return errMsg(`clientCertificates[${k}].ca[${k}]`, certsForUrl.ca[k], 'a relative filepath');
                }
            }
        }
    }
    return true;
};
exports.isValidClientCertificatesSet = isValidClientCertificatesSet;
const isPlainObject = (key, value) => {
    if (value == null || _.isPlainObject(value)) {
        return true;
    }
    return errMsg(key, value, 'a plain object');
};
exports.isPlainObject = isPlainObject;
function isBoolean(key, value) {
    if (value == null || _.isBoolean(value)) {
        return true;
    }
    return errMsg(key, value, 'a boolean');
}
exports.isBoolean = isBoolean;
function isNumber(key, value) {
    if (value == null || _.isNumber(value)) {
        return true;
    }
    return errMsg(key, value, 'a number');
}
exports.isNumber = isNumber;
function isString(key, value) {
    if (value == null || _.isString(value)) {
        return true;
    }
    return errMsg(key, value, 'a string');
}
exports.isString = isString;
function isArray(key, value) {
    if (value == null || _.isArray(value)) {
        return true;
    }
    return errMsg(key, value, 'an array');
}
exports.isArray = isArray;
function isNumberOrFalse(key, value) {
    if (_.isNumber(value) || isFalse(value)) {
        return true;
    }
    return errMsg(key, value, 'a number or false');
}
exports.isNumberOrFalse = isNumberOrFalse;
function isStringOrFalse(key, value) {
    if (_.isString(value) || isFalse(value)) {
        return true;
    }
    return errMsg(key, value, 'a string or false');
}
exports.isStringOrFalse = isStringOrFalse;
function isFullyQualifiedUrl(key, value) {
    if (value == null || _isFullyQualifiedUrl(value)) {
        return true;
    }
    return errMsg(key, value, 'a fully qualified URL (starting with `http://` or `https://`)');
}
exports.isFullyQualifiedUrl = isFullyQualifiedUrl;
function isStringOrArrayOfStrings(key, value) {
    if (_.isString(value) || isArrayOfStrings(value)) {
        return true;
    }
    return errMsg(key, value, 'a string or an array of strings');
}
exports.isStringOrArrayOfStrings = isStringOrArrayOfStrings;
