"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const nativeModule = process.platform === "win32"
    ? require("../../build/Release/registry.node")
    : null;
/**
 * Utility function used to achieve exhaustive type checks at compile time.
 *
 * If the type system is bypassed or this method will throw an exception
 * using the second parameter as the message.
 *
 * @param {x}       Placeholder parameter in order to leverage the type
 *                  system. Pass the variable which has been type narrowed
 *                  in an exhaustive check.
 *
 * @param {message} The message to be used in the runtime exception.
 *
 */
function assertNever(x, message) {
    throw new Error(message);
}
/**
 * Note: not all of these are currently implemented.
 *
 * Source: https://msdn.microsoft.com/en-us/library/windows/desktop/ms724884(v=vs.85).aspx
 */
var RegistryValueType;
(function (RegistryValueType) {
    RegistryValueType["REG_BINARY"] = "REG_BINARY";
    RegistryValueType["REG_DWORD"] = "REG_DWORD";
    RegistryValueType["REG_DWORD_LITTLE_ENDIAN"] = "REG_DWORD_LITTLE_ENDIAN";
    RegistryValueType["REG_DWORD_BIG_ENDIAN"] = "REG_DWORD_BIG_ENDIAN";
    RegistryValueType["REG_EXPAND_SZ"] = "REG_EXPAND_SZ";
    RegistryValueType["REG_LINK"] = "REG_LINK";
    RegistryValueType["REG_MULTI_SZ"] = "REG_MULTI_SZ";
    RegistryValueType["REG_NONE"] = "REG_NONE";
    RegistryValueType["REG_QWORD"] = "REG_QWORD";
    RegistryValueType["REG_QWORD_LITTLE_ENDIAN"] = "REG_QWORD_LITTLE_ENDIAN";
    RegistryValueType["REG_SZ"] = "REG_SZ";
})(RegistryValueType = exports.RegistryValueType || (exports.RegistryValueType = {}));
var HKEY;
(function (HKEY) {
    HKEY["HKEY_CLASSES_ROOT"] = "HKEY_CLASSES_ROOT";
    HKEY["HKEY_CURRENT_CONFIG"] = "HKEY_CURRENT_CONFIG";
    HKEY["HKEY_DYN_DATA"] = "HKEY_DYN_DATA";
    HKEY["HKEY_CURRENT_USER_LOCAL_SETTINGS"] = "HKEY_CURRENT_USER_LOCAL_SETTINGS";
    HKEY["HKEY_CURRENT_USER"] = "HKEY_CURRENT_USER";
    HKEY["HKEY_LOCAL_MACHINE"] = "HKEY_LOCAL_MACHINE";
    HKEY["HKEY_PERFORMANCE_DATA"] = "HKEY_PERFORMANCE_DATA";
    HKEY["HKEY_PERFORMANCE_TEXT"] = "HKEY_PERFORMANCE_TEXT";
    HKEY["HKEY_PERFORMANCE_NLSTEXT"] = "HKEY_PERFORMANCE_NLSTEXT";
    HKEY["HKEY_USERS"] = "HKEY_USERS";
})(HKEY = exports.HKEY || (exports.HKEY = {}));
function mapToLong(key) {
    if (key === HKEY.HKEY_CLASSES_ROOT)
        return 0x80000000;
    if (key === HKEY.HKEY_CURRENT_USER)
        return 0x80000001;
    if (key === HKEY.HKEY_LOCAL_MACHINE)
        return 0x80000002;
    if (key === HKEY.HKEY_USERS)
        return 0x80000003;
    if (key === HKEY.HKEY_PERFORMANCE_DATA)
        return 0x80000004;
    if (key === HKEY.HKEY_CURRENT_CONFIG)
        return 0x80000005;
    if (key === HKEY.HKEY_DYN_DATA)
        return 0x80000006;
    if (key === HKEY.HKEY_CURRENT_USER_LOCAL_SETTINGS)
        return 0x80000007;
    if (key === HKEY.HKEY_PERFORMANCE_TEXT)
        return 0x80000050;
    if (key === HKEY.HKEY_PERFORMANCE_NLSTEXT)
        return 0x80000060;
    return assertNever(key, "The key does not map to an expected number value");
}
function enumerateValues(key, subkey) {
    if (!nativeModule) {
        // this code is a no-op when the module is missing
        return [];
    }
    const hkey = mapToLong(key);
    const result = nativeModule.readValues(hkey, subkey);
    return result;
}
exports.enumerateValues = enumerateValues;
function enumerateValuesSafe(key, subkey) {
    try {
        return enumerateValues(key, subkey);
    }
    catch (_a) {
        return [];
    }
}
exports.enumerateValuesSafe = enumerateValuesSafe;
function enumerateKeys(key, subkey) {
    if (!nativeModule) {
        // this code is a no-op when the module is missing
        return [];
    }
    const hkey = mapToLong(key);
    const result = nativeModule.enumKeys(hkey, subkey);
    return result;
}
exports.enumerateKeys = enumerateKeys;
function enumerateKeysSafe(key, subkey) {
    try {
        return enumerateKeys(key, subkey);
    }
    catch (_a) {
        return [];
    }
}
exports.enumerateKeysSafe = enumerateKeysSafe;
//# sourceMappingURL=registry.js.map